const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');
const validator = require('validator');

const userSchema = new mongoose.Schema(
  {
    // Personal Information (FRS: Register Applicant)
    firstName: {
      type: String,
      required: [true, 'First name is required'],
      trim: true,
      minlength: [2, 'First name must be at least 2 characters'],
    },
    lastName: {
      type: String,
      required: [true, 'Last name is required'],
      trim: true,
      minlength: [2, 'Last name must be at least 2 characters'],
    },
    gender: {
      type: String,
      trim: true,
      enum: ['male', 'female', 'other'],
      required: [true, 'Gender is required'],
    },
    cell: {
      type: String,
      trim: true,
      required: [true, 'Cell phone number is required'],
      validate: [
        {
          validator: (value) => validator.isMobilePhone(value, 'any'),
          message: 'Please provide a valid phone number',
        },
      ],
    },
    dateOfBirth: {
      type: Date,
      required: [true, 'Date of birth is required'],
      validate: [validator.isDate, 'Please provide a valid date of birth'],
    },

    // Address (FRS: Applicant Address Details)
    address: {
      street: {
        type: String,
        required: [true, 'Street address is required'],
        trim: true,
      },
      city: {
        type: String,
        required: [true, 'City is required'],
        trim: true,
      },
      province: {
        type: String,
        required: [true, 'Province is required'],
        trim: true,
      },
      country: {
        type: String,
        required: [true, 'Country is required'],
        validate: [
          validator.isISO31661Alpha2,
          'Please provide a valid 2-letter country code',
        ],
        default: 'ZA', // Default to South Africa per FRS
      },
      zip: {
        type: String,
        required: [true, 'Postal code is required'],
        validate: [
          (value) => validator.isPostalCode(value, 'ZA'),
          'Please provide a valid South African postal code',
        ],
      },
    },

    // Authentication
    username: {
      type: String,
      required: [true, 'Username is required'],
      unique: true,
      trim: true,
      minlength: [4, 'Username must be at least 4 characters'],
    },
    email: {
      type: String,
      required: [true, 'Email is required'],
      unique: true,
      trim: true,
      lowercase: true,
      validate: [validator.isEmail, 'Please provide a valid email'],
    },
    password: {
      type: String,
      required: [true, 'Password is required'],
      minlength: [8, 'Password must be at least 8 characters'],
      select: false, // Prevent password from being returned in queries
    },
    role: {
      type: String,
      enum: ['systemAdmin', 'admin', 'student'],
      default: 'student',
      required: true,
    },
    accountStatus: {
      type: String,
      enum: ['active', 'inactive', 'suspended'],
      default: 'active',
      required: true,
    },
    isLocked: { type: Boolean, default: false },
    failedLoginAttempts: {
      type: Number,
      default: 0,
      max: [5, 'Account locked due to too many failed login attempts'],
    },
    lastLogin: { type: Date },

    // Academic Information (FRS: Bursary Application Conditions)
    academicDetails: {
      grade12Marks: {
        type: Number,
        required: ['Grade 12 marks are required for students'],
        min: [0, 'Marks cannot be negative'],
        max: [100, 'Marks cannot exceed 100'],
      },
      subjects: [
        {
          subject: String,
          mark: {
            type: Number,
            min: 0,
            max: 100,
          },
        },
      ],
      institution: {
        type: String,
        required: [ 'Institution name is required for students'],
        trim: true,
      },
      fieldOfStudy: {
        type: String,
        required: ['Field of study is required for students'],
        trim: true,
      },
      scarceSkill: {
        type: Boolean,
        default: false,
      },
    },

    // Financial Need (FRS: Bursary Application Conditions)
    financialNeed: {
      householdIncome: {
        type: Number,
        min: [0, 'Household income cannot be negative'],
      },
      dependants: {
        type: Number,
        min: [0, 'Number of dependants cannot be negative'],
      },
    },

    // Achievements (FRS: Artistic/Sports Capabilities)
    achievements: [
      {
        type: String,
        enum: ['academic', 'artistic', 'sports'],
        description: String,
      },
    ],

    // Password Reset
    resetPasswordToken: { type: String, select: false },
    resetPasswordExpires: { type: Date, select: false },

    // Email Verification
    isEmailVerified: { type: Boolean, default: false },
    emailVerificationToken: { type: String, select: false },
    emailVerificationExpires: { type: Date, select: false },

    // Audit Trail
    userActions: [
      {
        action: {
          type: String,
          enum: ['login', 'application_submit', 'profile_update', 'password_reset'],
        },
        timestamp: { type: Date, default: Date.now },
      },
    ],
  },
  {
    toJSON: {
      transform: (doc, ret) => {
        delete ret.password;
        delete ret.resetPasswordToken;
        delete ret.resetPasswordExpires;
        delete ret.emailVerificationToken;
        delete ret.emailVerificationExpires;
        return ret;
      },
    },
    timestamps: true,
  }
);

// Indexes
// userSchema.index({ email: 1 }, { unique: true, background: true });
// userSchema.index({ username: 1 }, { unique: true, background: true });
// userSchema.index({ role: 1, accountStatus: 1 }); // For common queries

// Password validation
userSchema.pre('save', async function (next) {
  if (this.isModified('password')) {
    if (!/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/.test(this.password)) {
      return next(new Error('Password must contain at least one uppercase letter, one lowercase letter, one number, and one special character'));
    }
    const salt = await bcrypt.genSalt(10);
    this.password = await bcrypt.hash(this.password, salt);
  }
  next();
});

// Conditional validation for student role
userSchema.pre('save', function (next) {
  if (this.role === 'student') {
    if (!this.academicDetails.grade12Marks) {
      return next(new Error('Grade 12 marks are required for students'));
    }
    if (!this.academicDetails.institution) {
      return next(new Error('Institution is required for students'));
    }
    if (!this.academicDetails.fieldOfStudy) {
      return next(new Error('Field of study is required for students'));
    }
  }
  next();
});

// Method to compare passwords
userSchema.methods.comparePassword = async function (candidatePassword) {
  return bcrypt.compare(candidatePassword, this.password);
};

// Method to generate verification token
userSchema.methods.generateToken = function (type) {
  const crypto = require('crypto');
  const token = crypto.randomBytes(32).toString('hex');
  if (type === 'resetPassword') {
    this.resetPasswordToken = token;
    this.resetPasswordExpires = Date.now() + 3600000; // 1 hour
  } else if (type === 'emailVerification') {
    this.emailVerificationToken = token;
    this.emailVerificationExpires = Date.now() + 24 * 3600000; // 24 hours
  }
  return token;
};

module.exports = mongoose.model('User', userSchema);